using IronPdf;
using ReportTests.Models;
using ReportTests.Templates;
using System.Collections.Concurrent;
using System.Text.Json;

namespace ReportTests;

class Program
{
    private static readonly string OutputDir = Path.Combine(Directory.GetCurrentDirectory(), "output");

    static async Task Main(string[] args)
    {
        Console.WriteLine("Crystal Reports Alternative - IronPDF Test Suite");
        Console.WriteLine("=".PadRight(50, '='));
        Console.WriteLine();

        // Load license key from multiple sources (priority order):
        // 1. Command line argument
        // 2. Environment variable
        // 3. appsettings.json file
        var licenseKey = GetLicenseKey(args);

        if (!string.IsNullOrEmpty(licenseKey))
        {
            License.LicenseKey = licenseKey;
            Console.WriteLine("License key applied successfully.");
        }
        else
        {
            Console.WriteLine("WARNING: No license key found.");
            Console.WriteLine("  Options:");
            Console.WriteLine("  1. Add to appsettings.json: {\"IronPdf\": {\"LicenseKey\": \"YOUR_KEY\"}}");
            Console.WriteLine("  2. Set IRONPDF_LICENSE_KEY environment variable");
            Console.WriteLine("  3. Pass --license=YOUR_KEY as argument");
            Console.WriteLine();
            Console.WriteLine("PDFs will have watermarks without a license.");
        }
        Console.WriteLine();

        // Create output directory
        Directory.CreateDirectory(OutputDir);
        Console.WriteLine($"Output directory: {OutputDir}");
        Console.WriteLine();

        var tests = new List<(string Name, Func<Task<bool>> Test)>
        {
            ("1. Quickstart Example", TestQuickstart),
            ("2. Invoice Generation", TestInvoiceGeneration),
            ("3. Text Headers and Footers", TestTextHeadersFooters),
            ("4. HTML Headers and Footers", TestHtmlHeadersFooters),
            ("5. Employee Directory", TestEmployeeDirectory),
            ("6. Sales Chart with JavaScript", TestSalesChart),
            ("7. Inventory with Conditional Formatting", TestInventoryReport),
            ("8. Combined/Merged Reports", TestCombinedReports),
            ("9. Table of Contents", TestTableOfContents),
            ("10. Batch Processing", TestBatchProcessing),
        };

        var results = new List<(string Name, bool Success, string? Error)>();

        foreach (var (name, test) in tests)
        {
            Console.WriteLine($"Running: {name}...");
            try
            {
                var success = await test();
                results.Add((name, success, null));
                Console.WriteLine(success ? "  ✓ PASSED" : "  ✗ FAILED");
            }
            catch (Exception ex)
            {
                results.Add((name, false, ex.Message));
                Console.WriteLine($"  ✗ ERROR: {ex.Message}");
            }
            Console.WriteLine();
        }

        // Summary
        Console.WriteLine("=".PadRight(50, '='));
        Console.WriteLine("TEST SUMMARY");
        Console.WriteLine("=".PadRight(50, '='));
        foreach (var (name, success, error) in results)
        {
            var status = success ? "✓ PASSED" : "✗ FAILED";
            Console.WriteLine($"  {status} - {name}");
            if (error != null)
                Console.WriteLine($"           Error: {error}");
        }

        var passed = results.Count(r => r.Success);
        var total = results.Count;
        Console.WriteLine();
        Console.WriteLine($"Results: {passed}/{total} tests passed");
        Console.WriteLine($"PDF outputs saved to: {OutputDir}");
    }

    /// <summary>
    /// Gets the IronPDF license key from various sources (command line, env var, or appsettings.json)
    /// </summary>
    static string? GetLicenseKey(string[] args)
    {
        // 1. Check command line argument
        if (args.Length > 0 && args[0].StartsWith("--license="))
        {
            return args[0].Substring("--license=".Length);
        }

        // 2. Check environment variable
        var envKey = Environment.GetEnvironmentVariable("IRONPDF_LICENSE_KEY");
        if (!string.IsNullOrEmpty(envKey))
        {
            return envKey;
        }

        // 3. Check appsettings.json
        var appSettingsPath = Path.Combine(AppContext.BaseDirectory, "appsettings.json");
        if (!File.Exists(appSettingsPath))
        {
            // Also check current directory (for development)
            appSettingsPath = Path.Combine(Directory.GetCurrentDirectory(), "appsettings.json");
        }

        if (File.Exists(appSettingsPath))
        {
            try
            {
                var json = File.ReadAllText(appSettingsPath);
                using var doc = JsonDocument.Parse(json);
                if (doc.RootElement.TryGetProperty("IronPdf", out var ironPdfSection) &&
                    ironPdfSection.TryGetProperty("LicenseKey", out var licenseKeyElement))
                {
                    return licenseKeyElement.GetString();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"Warning: Could not read appsettings.json: {ex.Message}");
            }
        }

        return null;
    }

    // Test 1: Quickstart Example
    static Task<bool> TestQuickstart()
    {
        var pdf = new ChromePdfRenderer()
            .RenderHtmlAsPdf("<h1>Sales Report</h1><table><tr><td>Q1</td><td>$50,000</td></tr></table>");

        var outputPath = Path.Combine(OutputDir, "01-quickstart.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 2: Invoice Generation
    static Task<bool> TestInvoiceGeneration()
    {
        var invoice = CreateSampleInvoice();
        var html = HtmlTemplates.GetInvoiceTemplate(invoice);

        var renderer = new ChromePdfRenderer();
        renderer.RenderingOptions.MarginTop = 10;
        renderer.RenderingOptions.MarginBottom = 10;
        renderer.RenderingOptions.MarginLeft = 10;
        renderer.RenderingOptions.MarginRight = 10;
        renderer.RenderingOptions.PaperSize = IronPdf.Rendering.PdfPaperSize.Letter;

        var pdf = renderer.RenderHtmlAsPdf(html);
        var outputPath = Path.Combine(OutputDir, "02-invoice.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 3: Text Headers and Footers
    static Task<bool> TestTextHeadersFooters()
    {
        var renderer = new ChromePdfRenderer();

        // Set starting page number
        renderer.RenderingOptions.FirstPageNumber = 1;

        // Add centered header with divider line
        renderer.RenderingOptions.TextHeader = new TextHeaderFooter
        {
            CenterText = "CONFIDENTIAL - Internal Use Only",
            DrawDividerLine = true,
            Font = IronSoftware.Drawing.FontTypes.Arial,
            FontSize = 10
        };

        // Add footer with date on left, page numbers on right
        renderer.RenderingOptions.TextFooter = new TextHeaderFooter
        {
            LeftText = "{date} {time}",
            RightText = "Page {page} of {total-pages}",
            DrawDividerLine = true,
            Font = IronSoftware.Drawing.FontTypes.Arial,
            FontSize = 9
        };

        // Set margins to accommodate header/footer
        renderer.RenderingOptions.MarginTop = 25;
        renderer.RenderingOptions.MarginBottom = 20;

        var html = "<h1>Document with Text Headers</h1><p>This document demonstrates text-based headers and footers.</p>" +
                   "<p>Lorem ipsum dolor sit amet, consectetur adipiscing elit.</p>".PadRight(2000, ' ');

        var pdf = renderer.RenderHtmlAsPdf(html);
        var outputPath = Path.Combine(OutputDir, "03-text-headers-footers.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 4: HTML Headers and Footers
    static Task<bool> TestHtmlHeadersFooters()
    {
        var renderer = new ChromePdfRenderer();

        // Configure HTML header with custom layout
        renderer.RenderingOptions.HtmlHeader = new HtmlHeaderFooter
        {
            MaxHeight = 30,
            HtmlFragment = @"
                <div style='display: flex; justify-content: space-between; align-items: center;
                            width: 100%; font-family: Arial; font-size: 10px; color: #666;'>
                    <span style='font-weight: bold; color: #3498db;'>ACME Corp</span>
                    <span>Invoice Report</span>
                    <span>{date}</span>
                </div>"
        };

        // Configure HTML footer with page info
        renderer.RenderingOptions.HtmlFooter = new HtmlHeaderFooter
        {
            MaxHeight = 20,
            HtmlFragment = @"
                <div style='text-align: center; font-size: 9px; color: #999;
                            border-top: 1px solid #ddd; padding-top: 5px;'>
                    Page {page} of {total-pages} | Generated on {date}
                </div>",
            DrawDividerLine = false
        };

        renderer.RenderingOptions.MarginTop = 35;
        renderer.RenderingOptions.MarginBottom = 25;

        var invoice = CreateSampleInvoice();
        var html = HtmlTemplates.GetInvoiceTemplate(invoice);
        var pdf = renderer.RenderHtmlAsPdf(html);
        var outputPath = Path.Combine(OutputDir, "04-html-headers-footers.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 5: Employee Directory
    static Task<bool> TestEmployeeDirectory()
    {
        var directory = CreateSampleEmployeeDirectory();
        var html = HtmlTemplates.GetEmployeeDirectoryTemplate(directory);

        var renderer = new ChromePdfRenderer();
        var pdf = renderer.RenderHtmlAsPdf(html);
        var outputPath = Path.Combine(OutputDir, "05-employee-directory.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 6: Sales Chart with JavaScript
    static Task<bool> TestSalesChart()
    {
        var salesData = new SalesReportModel
        {
            ReportTitle = "2026 Monthly Sales Report",
            MonthLabels = new[] { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" },
            MonthlySales = new[] { 45000m, 52000m, 48000m, 61000m, 55000m, 67000m, 72000m, 69000m, 58000m, 63000m, 71000m, 85000m }
        };

        var html = HtmlTemplates.GetSalesChartTemplate(salesData);

        var renderer = new ChromePdfRenderer();
        // Wait for JavaScript (Chart.js) to execute
        renderer.RenderingOptions.WaitFor.JavaScript(500);

        var pdf = renderer.RenderHtmlAsPdf(html);
        var outputPath = Path.Combine(OutputDir, "06-sales-chart.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 7: Inventory with Conditional Formatting
    static Task<bool> TestInventoryReport()
    {
        var inventory = CreateSampleInventory();
        var html = HtmlTemplates.GetInventoryTemplate(inventory);

        var renderer = new ChromePdfRenderer();
        var pdf = renderer.RenderHtmlAsPdf(html);
        var outputPath = Path.Combine(OutputDir, "07-inventory-report.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 8: Combined/Merged Reports
    static Task<bool> TestCombinedReports()
    {
        var renderer = new ChromePdfRenderer();

        // Generate sales report PDF
        var salesData = new SalesReportModel
        {
            ReportTitle = "Q4 Sales Summary",
            MonthLabels = new[] { "Oct", "Nov", "Dec" },
            MonthlySales = new[] { 63000m, 71000m, 85000m }
        };
        renderer.RenderingOptions.WaitFor.JavaScript(500);
        var salesPdf = renderer.RenderHtmlAsPdf(HtmlTemplates.GetSalesChartTemplate(salesData));

        // Generate inventory report PDF
        var inventoryPdf = renderer.RenderHtmlAsPdf(HtmlTemplates.GetInventoryTemplate(CreateSampleInventory()));

        // Merge PDFs into one document
        var combined = PdfDocument.Merge(salesPdf, inventoryPdf);
        var outputPath = Path.Combine(OutputDir, "08-combined-report.pdf");
        combined.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 9: Table of Contents
    static Task<bool> TestTableOfContents()
    {
        var html = HtmlTemplates.GetTableOfContentsTemplate();

        var renderer = new ChromePdfRenderer();
        renderer.RenderingOptions.TableOfContents = TableOfContentsTypes.WithPageNumbers;

        var pdf = renderer.RenderHtmlAsPdf(html);
        var outputPath = Path.Combine(OutputDir, "09-table-of-contents.pdf");
        pdf.SaveAs(outputPath);

        return Task.FromResult(File.Exists(outputPath) && new FileInfo(outputPath).Length > 0);
    }

    // Test 10: Batch Processing
    static async Task<bool> TestBatchProcessing()
    {
        var invoices = Enumerable.Range(1, 5).Select(i => new InvoiceModel
        {
            InvoiceNumber = $"INV-2026-{i:D3}",
            InvoiceDate = DateTime.Now.AddDays(-i),
            DueDate = DateTime.Now.AddDays(30 - i),
            Company = new CompanyInfo
            {
                Name = "Batch Test Corp",
                Address = "123 Test Street",
                City = "Test City, TC 12345",
                Phone = "(555) 123-4567",
                Email = "info@batchtest.com"
            },
            Customer = new CustomerInfo
            {
                Name = $"Customer {i}",
                Address = $"{i * 100} Customer Ave",
                City = "Customer City, CC 54321",
                Email = $"customer{i}@example.com"
            },
            Items = new List<LineItem>
            {
                new() { Description = $"Product A - Batch {i}", Quantity = i, UnitPrice = 100m },
                new() { Description = $"Product B - Batch {i}", Quantity = i * 2, UnitPrice = 50m }
            }
        }).ToList();

        var results = new ConcurrentBag<ReportResult>();

        await Parallel.ForEachAsync(invoices,
            new ParallelOptions { MaxDegreeOfParallelism = Environment.ProcessorCount },
            async (invoice, token) =>
            {
                var renderer = new ChromePdfRenderer();
                var html = HtmlTemplates.GetInvoiceTemplate(invoice);
                var pdf = await renderer.RenderHtmlAsPdfAsync(html);

                var filename = Path.Combine(OutputDir, $"10-batch-{invoice.InvoiceNumber}.pdf");
                pdf.SaveAs(filename);

                results.Add(new ReportResult { InvoiceNumber = invoice.InvoiceNumber, Success = true });
            });

        return results.Count == 5 && results.All(r => r.Success);
    }

    // Helper methods to create sample data
    static InvoiceModel CreateSampleInvoice()
    {
        return new InvoiceModel
        {
            InvoiceNumber = "INV-2026-001",
            InvoiceDate = DateTime.Now,
            DueDate = DateTime.Now.AddDays(30),
            Company = new CompanyInfo
            {
                Name = "Acme Corporation",
                Address = "123 Business Street",
                City = "New York, NY 10001",
                Phone = "(555) 123-4567",
                Email = "billing@acmecorp.com"
            },
            Customer = new CustomerInfo
            {
                Name = "John Smith",
                Address = "456 Customer Lane",
                City = "Los Angeles, CA 90001",
                Email = "john.smith@example.com"
            },
            Items = new List<LineItem>
            {
                new() { Description = "Web Development Services", Quantity = 40, UnitPrice = 150m },
                new() { Description = "UI/UX Design", Quantity = 20, UnitPrice = 125m },
                new() { Description = "Server Hosting (Monthly)", Quantity = 1, UnitPrice = 99m },
                new() { Description = "SSL Certificate", Quantity = 1, UnitPrice = 49m },
                new() { Description = "Technical Support Hours", Quantity = 10, UnitPrice = 75m }
            }
        };
    }

    static EmployeeDirectoryModel CreateSampleEmployeeDirectory()
    {
        return new EmployeeDirectoryModel
        {
            Departments = new List<Department>
            {
                new()
                {
                    Name = "Engineering",
                    ManagerName = "Sarah Johnson",
                    Employees = new List<Employee>
                    {
                        new() { Name = "Mike Chen", Title = "Senior Developer", Email = "mike.chen@company.com", Phone = "(555) 101-0001", HireDate = new DateTime(2020, 3, 15) },
                        new() { Name = "Emily Davis", Title = "Full Stack Developer", Email = "emily.davis@company.com", Phone = "(555) 101-0002", HireDate = new DateTime(2021, 7, 1) },
                        new() { Name = "Alex Rivera", Title = "DevOps Engineer", Email = "alex.rivera@company.com", Phone = "(555) 101-0003", HireDate = new DateTime(2022, 1, 10) }
                    }
                },
                new()
                {
                    Name = "Marketing",
                    ManagerName = "David Wilson",
                    Employees = new List<Employee>
                    {
                        new() { Name = "Jessica Brown", Title = "Marketing Manager", Email = "jessica.brown@company.com", Phone = "(555) 102-0001", HireDate = new DateTime(2019, 5, 20) },
                        new() { Name = "Chris Taylor", Title = "Content Strategist", Email = "chris.taylor@company.com", Phone = "(555) 102-0002", HireDate = new DateTime(2023, 2, 14) }
                    }
                },
                new()
                {
                    Name = "Sales",
                    ManagerName = "Patricia Martinez",
                    Employees = new List<Employee>
                    {
                        new() { Name = "Robert Kim", Title = "Sales Representative", Email = "robert.kim@company.com", Phone = "(555) 103-0001", HireDate = new DateTime(2021, 9, 1) },
                        new() { Name = "Amanda Lee", Title = "Account Executive", Email = "amanda.lee@company.com", Phone = "(555) 103-0002", HireDate = new DateTime(2022, 4, 18) },
                        new() { Name = "James Garcia", Title = "Sales Analyst", Email = "james.garcia@company.com", Phone = "(555) 103-0003", HireDate = new DateTime(2023, 6, 5) }
                    }
                }
            }
        };
    }

    static InventoryReportModel CreateSampleInventory()
    {
        return new InventoryReportModel
        {
            LowStockThreshold = 20,
            CriticalStockThreshold = 5,
            Items = new List<InventoryItem>
            {
                new() { SKU = "ELEC-001", ProductName = "Wireless Mouse", Category = "Electronics", Quantity = 150, UnitCost = 25.99m },
                new() { SKU = "ELEC-002", ProductName = "USB-C Hub", Category = "Electronics", Quantity = 3, UnitCost = 45.00m },
                new() { SKU = "ELEC-003", ProductName = "Mechanical Keyboard", Category = "Electronics", Quantity = 12, UnitCost = 89.99m },
                new() { SKU = "OFF-001", ProductName = "Printer Paper (Ream)", Category = "Office Supplies", Quantity = 500, UnitCost = 8.99m },
                new() { SKU = "OFF-002", ProductName = "Sticky Notes Pack", Category = "Office Supplies", Quantity = 18, UnitCost = 4.99m },
                new() { SKU = "OFF-003", ProductName = "Whiteboard Markers", Category = "Office Supplies", Quantity = 2, UnitCost = 12.99m },
                new() { SKU = "FURN-001", ProductName = "Office Chair", Category = "Furniture", Quantity = 25, UnitCost = 199.00m },
                new() { SKU = "FURN-002", ProductName = "Standing Desk", Category = "Furniture", Quantity = 8, UnitCost = 449.00m },
                new() { SKU = "TECH-001", ProductName = "Monitor 27\"", Category = "Technology", Quantity = 45, UnitCost = 299.99m },
                new() { SKU = "TECH-002", ProductName = "Webcam HD", Category = "Technology", Quantity = 5, UnitCost = 79.99m }
            }
        };
    }
}
