using ReportTests.Models;

namespace ReportTests.Templates;

public static class HtmlTemplates
{
    public static string GetInvoiceTemplate(InvoiceModel model)
    {
        var itemRows = string.Join("\n", model.Items.Select(item => $@"
            <tr>
                <td>{item.Description}</td>
                <td>{item.Quantity}</td>
                <td>{item.UnitPrice:C}</td>
                <td>{item.Total:C}</td>
            </tr>"));

        return $@"
<!DOCTYPE html>
<html>
<head>
    <meta charset=""UTF-8"">
    <style>
        /* Reset and base styles */
        * {{ margin: 0; padding: 0; box-sizing: border-box; }}
        body {{ font-family: 'Segoe UI', Arial, sans-serif; font-size: 12px; color: #333; line-height: 1.5; }}
        .invoice-container {{ max-width: 800px; margin: 0 auto; padding: 40px; }}

        /* Header with company info and invoice title */
        .header {{ display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 40px; padding-bottom: 20px; border-bottom: 2px solid #3498db; }}
        .company-info h1 {{ font-size: 24px; color: #2c3e50; margin-bottom: 10px; }}
        .company-info p {{ color: #7f8c8d; font-size: 11px; }}
        .invoice-title {{ text-align: right; }}
        .invoice-title h2 {{ font-size: 32px; color: #3498db; margin-bottom: 10px; }}
        .invoice-title p {{ font-size: 12px; color: #7f8c8d; }}

        /* Address blocks */
        .addresses {{ display: flex; justify-content: space-between; margin-bottom: 30px; }}
        .address-block {{ width: 45%; }}
        .address-block h3 {{ font-size: 11px; text-transform: uppercase; color: #95a5a6; margin-bottom: 8px; letter-spacing: 1px; }}
        .address-block p {{ font-size: 12px; }}

        /* Line items table */
        .items-table {{ width: 100%; border-collapse: collapse; margin-bottom: 30px; }}
        .items-table th {{ background-color: #3498db; color: white; padding: 12px; text-align: left; font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px; }}
        .items-table th:last-child, .items-table td:last-child {{ text-align: right; }}
        .items-table td {{ padding: 12px; border-bottom: 1px solid #ecf0f1; }}
        .items-table tr:nth-child(even) {{ background-color: #f9f9f9; }}

        /* Totals section */
        .totals {{ float: right; width: 300px; }}
        .totals-row {{ display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #ecf0f1; }}
        .totals-row.grand-total {{ border-bottom: none; border-top: 2px solid #3498db; font-size: 16px; font-weight: bold; color: #2c3e50; padding-top: 12px; }}

        /* Footer */
        .footer {{ clear: both; margin-top: 60px; padding-top: 20px; border-top: 1px solid #ecf0f1; text-align: center; color: #95a5a6; font-size: 10px; }}
    </style>
</head>
<body>
    <div class=""invoice-container"">
        <!-- Company header and invoice details -->
        <div class=""header"">
            <div class=""company-info"">
                <h1>{model.Company.Name}</h1>
                <p>{model.Company.Address}</p>
                <p>{model.Company.City}</p>
                <p>{model.Company.Phone} | {model.Company.Email}</p>
            </div>
            <div class=""invoice-title"">
                <h2>INVOICE</h2>
                <p><strong>Invoice #:</strong> {model.InvoiceNumber}</p>
                <p><strong>Date:</strong> {model.InvoiceDate:MMMM dd, yyyy}</p>
                <p><strong>Due Date:</strong> {model.DueDate:MMMM dd, yyyy}</p>
            </div>
        </div>

        <!-- Bill to address -->
        <div class=""addresses"">
            <div class=""address-block"">
                <h3>Bill To</h3>
                <p><strong>{model.Customer.Name}</strong></p>
                <p>{model.Customer.Address}</p>
                <p>{model.Customer.City}</p>
                <p>{model.Customer.Email}</p>
            </div>
        </div>

        <!-- Line items table -->
        <table class=""items-table"">
            <thead>
                <tr><th>Description</th><th>Quantity</th><th>Unit Price</th><th>Total</th></tr>
            </thead>
            <tbody>
                {itemRows}
            </tbody>
        </table>

        <!-- Totals -->
        <div class=""totals"">
            <div class=""totals-row""><span>Subtotal:</span><span>{model.Subtotal:C}</span></div>
            <div class=""totals-row""><span>Tax ({model.TaxRate * 100}%):</span><span>{model.TaxAmount:C}</span></div>
            <div class=""totals-row grand-total""><span>Total Due:</span><span>{model.GrandTotal:C}</span></div>
        </div>

        <!-- Footer message -->
        <div class=""footer"">
            <p>Thank you for your business!</p>
            <p>Payment is due within 30 days. Please include invoice number with your payment.</p>
        </div>
    </div>
</body>
</html>";
    }

    public static string GetEmployeeDirectoryTemplate(EmployeeDirectoryModel model)
    {
        var departmentSections = string.Join("\n", model.Departments.Select(dept =>
        {
            var employeeRows = string.Join("\n", dept.Employees.Select(emp => $@"
                <tr>
                    <td>{emp.Name}</td>
                    <td>{emp.Title}</td>
                    <td>{emp.Email}</td>
                    <td>{emp.Phone}</td>
                    <td>{emp.HireDate:MMM dd, yyyy}</td>
                </tr>"));

            return $@"
            <div class=""department"">
                <h2>{dept.Name}</h2>
                <p class=""manager"">Manager: {dept.ManagerName}</p>
                <table class=""employee-table"">
                    <thead>
                        <tr><th>Name</th><th>Title</th><th>Email</th><th>Phone</th><th>Hire Date</th></tr>
                    </thead>
                    <tbody>
                        {employeeRows}
                    </tbody>
                </table>
            </div>";
        }));

        return $@"
<!DOCTYPE html>
<html>
<head>
    <meta charset=""UTF-8"">
    <style>
        * {{ margin: 0; padding: 0; box-sizing: border-box; }}
        body {{ font-family: 'Segoe UI', Arial, sans-serif; font-size: 12px; color: #333; padding: 40px; }}
        h1 {{ color: #2c3e50; margin-bottom: 20px; }}
        .generated {{ color: #7f8c8d; margin-bottom: 30px; }}
        .department {{ margin-bottom: 30px; page-break-inside: avoid; }}
        .department:not(:first-child) {{ page-break-before: always; }}
        .department h2 {{ color: #3498db; border-bottom: 2px solid #3498db; padding-bottom: 10px; margin-bottom: 10px; }}
        .manager {{ color: #7f8c8d; margin-bottom: 15px; font-style: italic; }}
        .employee-table {{ width: 100%; border-collapse: collapse; }}
        .employee-table th {{ background-color: #ecf0f1; padding: 10px; text-align: left; font-size: 11px; text-transform: uppercase; }}
        .employee-table td {{ padding: 10px; border-bottom: 1px solid #ecf0f1; }}
        .employee-table tr:nth-child(even) {{ background-color: #f9f9f9; }}
    </style>
</head>
<body>
    <h1>Employee Directory</h1>
    <p class=""generated"">Generated: {model.GeneratedDate:MMMM dd, yyyy}</p>
    {departmentSections}
</body>
</html>";
    }

    public static string GetInventoryTemplate(InventoryReportModel model)
    {
        var itemRows = string.Join("\n", model.Items.OrderBy(x => x.Quantity).Select(item =>
        {
            var rowClass = item.Quantity <= model.CriticalStockThreshold ? "stock-critical" :
                           item.Quantity <= model.LowStockThreshold ? "stock-low" : "";
            var badgeClass = item.Quantity <= 5 ? "badge-critical" : "badge-ok";

            return $@"
            <tr class=""{rowClass}"">
                <td>{item.SKU}</td>
                <td>{item.ProductName}</td>
                <td>{item.Category}</td>
                <td class=""text-right"">
                    <span class=""quantity-badge {badgeClass}"">{item.Quantity}</span>
                </td>
                <td class=""text-right"">{item.UnitCost:C}</td>
            </tr>";
        }));

        return $@"
<!DOCTYPE html>
<html>
<head>
    <meta charset=""UTF-8"">
    <style>
        * {{ margin: 0; padding: 0; box-sizing: border-box; }}
        body {{ font-family: 'Segoe UI', Arial, sans-serif; font-size: 12px; color: #333; padding: 40px; }}
        h1 {{ color: #2c3e50; margin-bottom: 10px; }}
        .subtitle {{ color: #7f8c8d; margin-bottom: 20px; }}
        .legend {{ margin-bottom: 20px; }}
        .legend span {{ padding: 3px 10px; margin-right: 10px; border-radius: 3px; font-size: 11px; }}
        .legend .critical {{ background-color: #e74c3c; color: white; }}
        .legend .low {{ background-color: #f39c12; color: white; }}
        .legend .ok {{ background-color: #27ae60; color: white; }}
        table {{ width: 100%; border-collapse: collapse; }}
        th {{ background-color: #34495e; color: white; padding: 12px; text-align: left; font-size: 11px; text-transform: uppercase; }}
        td {{ padding: 12px; border-bottom: 1px solid #ecf0f1; }}
        .text-right {{ text-align: right; }}
        .stock-critical {{ background-color: #fadbd8; }}
        .stock-low {{ background-color: #fef9e7; }}
        .quantity-badge {{ padding: 3px 8px; border-radius: 3px; font-weight: bold; }}
        .badge-critical {{ background-color: #e74c3c; color: white; }}
        .badge-ok {{ background-color: #27ae60; color: white; }}
    </style>
</head>
<body>
    <h1>Inventory Report</h1>
    <p class=""subtitle"">Generated: {model.GeneratedDate:MMMM dd, yyyy} | Critical: ≤{model.CriticalStockThreshold} | Low: ≤{model.LowStockThreshold}</p>
    <div class=""legend"">
        <span class=""critical"">Critical Stock</span>
        <span class=""low"">Low Stock</span>
        <span class=""ok"">OK</span>
    </div>
    <table>
        <thead>
            <tr><th>SKU</th><th>Product Name</th><th>Category</th><th class=""text-right"">Quantity</th><th class=""text-right"">Unit Cost</th></tr>
        </thead>
        <tbody>
            {itemRows}
        </tbody>
    </table>
</body>
</html>";
    }

    public static string GetSalesChartTemplate(SalesReportModel model)
    {
        var labelsJson = System.Text.Json.JsonSerializer.Serialize(model.MonthLabels);
        var dataJson = System.Text.Json.JsonSerializer.Serialize(model.MonthlySales);

        return $@"
<!DOCTYPE html>
<html>
<head>
    <meta charset=""UTF-8"">
    <script src=""https://cdn.jsdelivr.net/npm/chart.js""></script>
    <style>
        * {{ margin: 0; padding: 0; box-sizing: border-box; }}
        body {{ font-family: 'Segoe UI', Arial, sans-serif; padding: 40px; }}
        h1 {{ color: #2c3e50; margin-bottom: 10px; }}
        .subtitle {{ color: #7f8c8d; margin-bottom: 30px; }}
        .chart-container {{ max-width: 700px; margin: 0 auto; }}
    </style>
</head>
<body>
    <h1>{model.ReportTitle}</h1>
    <p class=""subtitle"">Generated: {model.GeneratedDate:MMMM dd, yyyy}</p>
    <div class=""chart-container"">
        <canvas id=""salesChart""></canvas>
    </div>
    <script>
        const ctx = document.getElementById('salesChart').getContext('2d');
        new Chart(ctx, {{
            type: 'bar',
            data: {{
                labels: {labelsJson},
                datasets: [{{
                    label: 'Monthly Sales ($)',
                    data: {dataJson},
                    backgroundColor: 'rgba(52, 152, 219, 0.7)',
                    borderColor: 'rgba(52, 152, 219, 1)',
                    borderWidth: 1
                }}]
            }},
            options: {{
                responsive: true,
                scales: {{
                    y: {{
                        beginAtZero: true,
                        ticks: {{
                            callback: function(value) {{ return '$' + value.toLocaleString(); }}
                        }}
                    }}
                }}
            }}
        }});
    </script>
</body>
</html>";
    }

    public static string GetTableOfContentsTemplate()
    {
        return @"
<!DOCTYPE html>
<html>
<head>
    <meta charset=""UTF-8"">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Arial, sans-serif; padding: 40px; }
        h1 { color: #2c3e50; margin-bottom: 30px; border-bottom: 2px solid #3498db; padding-bottom: 10px; }
        h2 { color: #3498db; margin: 30px 0 15px 0; }
        h3 { color: #7f8c8d; margin: 20px 0 10px 20px; }
        p { margin-bottom: 10px; line-height: 1.6; }
        .section { margin-bottom: 30px; }
    </style>
</head>
<body>
    <h1>Annual Report 2026</h1>

    <div class=""section"">
        <h2>Executive Summary</h2>
        <p>This section provides an overview of the company's performance throughout the year, including key achievements and milestones.</p>
        <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.</p>
    </div>

    <div class=""section"">
        <h2>Financial Overview</h2>
        <h3>Revenue Analysis</h3>
        <p>Detailed breakdown of revenue streams and growth patterns observed during the fiscal year.</p>
        <h3>Cost Management</h3>
        <p>Overview of cost optimization initiatives and their impact on overall profitability.</p>
    </div>

    <div class=""section"">
        <h2>Operations Report</h2>
        <h3>Production Metrics</h3>
        <p>Analysis of production efficiency and quality metrics across all facilities.</p>
        <h3>Supply Chain Performance</h3>
        <p>Evaluation of supply chain resilience and vendor relationship management.</p>
    </div>

    <div class=""section"">
        <h2>Future Outlook</h2>
        <p>Strategic initiatives planned for the upcoming year and long-term growth projections.</p>
    </div>
</body>
</html>";
    }
}
