using System;
using System.Diagnostics;
using System.IO;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using IronPdf;
using IronPdf.Signing;

namespace PdfDigitalSignaturesTest
{
    class Program
    {
        // Test output directory
        static readonly string OutputDir = Path.Combine(Directory.GetCurrentDirectory(), "output");
        static readonly string CertificatePath = Path.Combine(Directory.GetCurrentDirectory(), "test-certificate.pfx");
        static readonly string CertificatePassword = "testpassword123";

        static void Main(string[] args)
        {
            Console.WriteLine("=== PDF Digital Signatures Test Project ===\n");

            // Ensure output directory exists
            Directory.CreateDirectory(OutputDir);

            // Create test certificate if it doesn't exist
            if (!File.Exists(CertificatePath))
            {
                Console.WriteLine("Creating test certificate using openssl...");
                bool created = CreateTestCertificateWithOpenSSL();
                if (!created)
                {
                    Console.WriteLine("OpenSSL not available, creating with .NET...");
                    CreateTestCertificateWithDotNet();
                }
                Console.WriteLine("Test certificate created successfully.\n");
            }

            // Run all tests
            try
            {
                Test1_LoadCertificate();
                Test2_BasicSigning();
                Test3_OneLineSigning();
                Test4_SignatureMetadata();
                Test5_TimestampServer();
                Test6_VisualSignature();
                Test7_MultiPageSigning();
                Test8_SequentialSigning();
                Test9_VerifySignatures();
                Test10_DetectTampering();
                Test11_ComplianceSigning();

                // Generate PNG screenshots of input PDFs for documentation
                GenerateScreenshots();

                Console.WriteLine("\n=== All Tests Completed ===");
                Console.WriteLine($"Output files saved to: {OutputDir}");
            }
            catch (Exception ex)
            {
                Console.WriteLine($"\nError: {ex.Message}");
                Console.WriteLine(ex.StackTrace);
            }
        }

        static bool CreateTestCertificateWithOpenSSL()
        {
            try
            {
                var startInfo = new ProcessStartInfo
                {
                    FileName = "openssl",
                    Arguments = $"req -x509 -newkey rsa:2048 -keyout key.pem -out cert.pem -days 365 -nodes -subj \"/CN=Test Signer/O=Test Organization/C=US\"",
                    WorkingDirectory = Directory.GetCurrentDirectory(),
                    UseShellExecute = false,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true,
                    CreateNoWindow = true
                };

                using var process = Process.Start(startInfo);
                process?.WaitForExit(30000);

                if (process?.ExitCode == 0)
                {
                    // Convert to PFX
                    var pfxStartInfo = new ProcessStartInfo
                    {
                        FileName = "openssl",
                        Arguments = $"pkcs12 -export -out \"{CertificatePath}\" -inkey key.pem -in cert.pem -password pass:{CertificatePassword}",
                        WorkingDirectory = Directory.GetCurrentDirectory(),
                        UseShellExecute = false,
                        RedirectStandardOutput = true,
                        RedirectStandardError = true,
                        CreateNoWindow = true
                    };

                    using var pfxProcess = Process.Start(pfxStartInfo);
                    pfxProcess?.WaitForExit(30000);

                    // Clean up temp files
                    File.Delete(Path.Combine(Directory.GetCurrentDirectory(), "key.pem"));
                    File.Delete(Path.Combine(Directory.GetCurrentDirectory(), "cert.pem"));

                    return pfxProcess?.ExitCode == 0 && File.Exists(CertificatePath);
                }
            }
            catch
            {
                // OpenSSL not available
            }
            return false;
        }

        static void CreateTestCertificateWithDotNet()
        {
            // Create a self-signed certificate for testing
            using var rsa = RSA.Create(2048);
            var request = new CertificateRequest(
                "CN=Test Signer, O=Test Organization, C=US",
                rsa,
                HashAlgorithmName.SHA256,
                RSASignaturePadding.Pkcs1);

            request.CertificateExtensions.Add(
                new X509KeyUsageExtension(
                    X509KeyUsageFlags.DigitalSignature | X509KeyUsageFlags.NonRepudiation,
                    critical: true));

            var certificate = request.CreateSelfSigned(
                DateTimeOffset.Now,
                DateTimeOffset.Now.AddYears(1));

            // Export to PFX bytes
            var pfxBytes = certificate.Export(X509ContentType.Pfx, CertificatePassword);

            // Write the certificate
            File.WriteAllBytes(CertificatePath, pfxBytes);
        }

        static PdfDocument CreateSamplePdf(string title, string content)
        {
            var html = $@"
            <html>
            <head>
                <style>
                    body {{ font-family: Arial, sans-serif; padding: 40px; }}
                    h1 {{ color: #333; }}
                    .content {{ margin-top: 20px; line-height: 1.6; }}
                    .signature-area {{ margin-top: 100px; border-top: 1px solid #ccc; padding-top: 20px; }}
                </style>
            </head>
            <body>
                <h1>{title}</h1>
                <div class='content'>{content}</div>
                <div class='signature-area'>
                    <p><strong>Signature:</strong> _______________________</p>
                    <p><strong>Date:</strong> _______________________</p>
                </div>
            </body>
            </html>";

            var renderer = new ChromePdfRenderer();
            return renderer.RenderHtmlAsPdf(html);
        }

        static void Test1_LoadCertificate()
        {
            Console.WriteLine("Test 1: Load Certificate");
            Console.WriteLine("------------------------");

            var certificate = new X509Certificate2(
                CertificatePath,
                CertificatePassword,
                X509KeyStorageFlags.Exportable);

            Console.WriteLine("Certificate loaded successfully");
            Console.WriteLine($"  Subject: {certificate.Subject}");
            Console.WriteLine($"  Issuer: {certificate.Issuer}");
            Console.WriteLine($"  Valid From: {certificate.NotBefore}");
            Console.WriteLine($"  Valid To: {certificate.NotAfter}");
            Console.WriteLine($"  Thumbprint: {certificate.Thumbprint}");
            Console.WriteLine($"  Has Private Key: {certificate.HasPrivateKey}");
            Console.WriteLine();
        }

        static void Test2_BasicSigning()
        {
            Console.WriteLine("Test 2: Basic Certificate Signing");
            Console.WriteLine("----------------------------------");

            // Create a sample PDF
            var pdf = CreateSamplePdf("Sample Contract",
                "This is a sample contract document that will be digitally signed.");
            var inputPath = Path.Combine(OutputDir, "contract-input.pdf");
            pdf.SaveAs(inputPath);
            Console.WriteLine($"  Created: contract-input.pdf");

            // Load and sign
            var pdfToSign = PdfDocument.FromFile(inputPath);
            var signature = new PdfSignature(CertificatePath, CertificatePassword);
            pdfToSign.Sign(signature);

            var outputPath = Path.Combine(OutputDir, "contract-signed.pdf");
            pdfToSign.SaveAs(outputPath);
            Console.WriteLine($"  Created: contract-signed.pdf");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test3_OneLineSigning()
        {
            Console.WriteLine("Test 3: One-Line Signing");
            Console.WriteLine("------------------------");

            // Create a sample PDF first
            var pdf = CreateSamplePdf("Quick Document", "This document uses one-line signing.");
            var inputPath = Path.Combine(OutputDir, "quick-document.pdf");
            pdf.SaveAs(inputPath);

            // One-line signing approach
            var signature = new PdfSignature(CertificatePath, CertificatePassword);
            PdfDocument.FromFile(inputPath).Sign(signature).SaveAs(Path.Combine(OutputDir, "quick-document-signed.pdf"));

            Console.WriteLine($"  Created: quick-document-signed.pdf");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test4_SignatureMetadata()
        {
            Console.WriteLine("Test 4: Signature Metadata");
            Console.WriteLine("--------------------------");

            var pdf = CreateSamplePdf("Invoice #12345",
                "Invoice for services rendered. Amount: $1,500.00");
            var inputPath = Path.Combine(OutputDir, "invoice-input.pdf");
            pdf.SaveAs(inputPath);
            Console.WriteLine($"  Created: invoice-input.pdf");

            var pdfToSign = PdfDocument.FromFile(inputPath);
            var signature = new PdfSignature(CertificatePath, CertificatePassword)
            {
                SigningReason = "Invoice Approval",
                SigningLocation = "New York Office",
                SigningContact = "accounts@company.com",
                SignatureDate = DateTime.UtcNow
            };

            pdfToSign.Sign(signature);
            var outputPath = Path.Combine(OutputDir, "invoice-approved.pdf");
            pdfToSign.SaveAs(outputPath);

            Console.WriteLine($"  Created: invoice-approved.pdf");
            Console.WriteLine($"  Metadata - Reason: {signature.SigningReason}");
            Console.WriteLine($"  Metadata - Location: {signature.SigningLocation}");
            Console.WriteLine($"  Metadata - Contact: {signature.SigningContact}");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test5_TimestampServer()
        {
            Console.WriteLine("Test 5: Timestamp Server");
            Console.WriteLine("------------------------");

            var pdf = CreateSamplePdf("Service Agreement",
                "This agreement is entered into between Party A and Party B.");
            var inputPath = Path.Combine(OutputDir, "agreement-input.pdf");
            pdf.SaveAs(inputPath);
            Console.WriteLine($"  Created: agreement-input.pdf");

            var pdfToSign = PdfDocument.FromFile(inputPath);
            var signature = new PdfSignature(CertificatePath, CertificatePassword)
            {
                SigningReason = "Agreement Execution",
                // Note: Using a public timestamp server
                // TimestampHashAlgorithm = TimestampHashAlgorithms.SHA256,
                // TimeStampUrl = new Uri("http://timestamp.digicert.com")
            };

            pdfToSign.Sign(signature);
            var outputPath = Path.Combine(OutputDir, "agreement-timestamped.pdf");
            pdfToSign.SaveAs(outputPath);

            Console.WriteLine($"  Created: agreement-timestamped.pdf");
            Console.WriteLine("  Note: Timestamp server commented out for offline testing");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test6_VisualSignature()
        {
            Console.WriteLine("Test 6: Visual Signature");
            Console.WriteLine("------------------------");

            var pdf = CreateSamplePdf("Contract with Visual Signature",
                "This contract includes a visible signature block for visual confirmation.");
            var inputPath = Path.Combine(OutputDir, "contract-visual-input.pdf");
            pdf.SaveAs(inputPath);
            Console.WriteLine($"  Created: contract-visual-input.pdf");

            // Create a simple signature image
            var signatureImagePath = Path.Combine(OutputDir, "signature-image.png");
            CreateSignatureImage(signatureImagePath);
            Console.WriteLine($"  Created: signature-image.png");

            var pdfToSign = PdfDocument.FromFile(inputPath);
            var signature = new PdfSignature(CertificatePath, CertificatePassword)
            {
                SigningReason = "Contract Approval",
                SigningLocation = "Head Office"
            };

            // Add visual signature image using LoadSignatureImageFromFile
            signature.LoadSignatureImageFromFile(signatureImagePath, 0, new IronSoftware.Drawing.Rectangle(150, 100, 200, 50));

            pdfToSign.Sign(signature);
            var outputPath = Path.Combine(OutputDir, "contract-visually-signed.pdf");
            pdfToSign.SaveAs(outputPath);

            Console.WriteLine($"  Created: contract-visually-signed.pdf");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void CreateSignatureImage(string path)
        {
            // Create a simple signature image using HTML rendering
            var html = @"
            <div style='font-family: ""Brush Script MT"", cursive; font-size: 32px; color: #000080;
                        padding: 5px; background: white;'>
                John Smith
            </div>";

            var renderer = new ChromePdfRenderer();
            renderer.RenderingOptions.PaperSize = IronPdf.Rendering.PdfPaperSize.Custom;
            renderer.RenderingOptions.SetCustomPaperSizeinPixelsOrPoints(250, 80);
            renderer.RenderingOptions.MarginTop = 0;
            renderer.RenderingOptions.MarginBottom = 0;
            renderer.RenderingOptions.MarginLeft = 0;
            renderer.RenderingOptions.MarginRight = 0;

            var tempPdf = renderer.RenderHtmlAsPdf(html);
            tempPdf.RasterizeToImageFiles(path);
        }

        static void Test7_MultiPageSigning()
        {
            Console.WriteLine("Test 7: Multi-Page Signing");
            Console.WriteLine("--------------------------");

            // Create multi-page PDF
            var html = @"
            <html>
            <head><style>
                body { font-family: Arial; padding: 40px; }
                .page { page-break-after: always; min-height: 90vh; }
                .page:last-child { page-break-after: avoid; }
            </style></head>
            <body>
                <div class='page'><h1>Page 1: Introduction</h1><p>Introduction content...</p></div>
                <div class='page'><h1>Page 2: Terms</h1><p>Terms and conditions...</p></div>
                <div class='page'><h1>Page 3: Signature Page</h1><p>Please sign below:</p>
                    <div style='margin-top: 200px;'>Signature: _______________</div>
                </div>
            </body>
            </html>";

            var renderer = new ChromePdfRenderer();
            var pdf = renderer.RenderHtmlAsPdf(html);
            var inputPath = Path.Combine(OutputDir, "multi-page-contract-input.pdf");
            pdf.SaveAs(inputPath);
            Console.WriteLine($"  Created: multi-page-contract-input.pdf ({pdf.PageCount} pages)");

            var pdfToSign = PdfDocument.FromFile(inputPath);
            var signature = new PdfSignature(CertificatePath, CertificatePassword);

            pdfToSign.Sign(signature);
            var outputPath = Path.Combine(OutputDir, "contract-signed-last-page.pdf");
            pdfToSign.SaveAs(outputPath);

            Console.WriteLine($"  Created: contract-signed-last-page.pdf");
            Console.WriteLine($"  Signature applied to document with {pdfToSign.PageCount} pages");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test8_SequentialSigning()
        {
            Console.WriteLine("Test 8: Sequential Signing (Multi-Party)");
            Console.WriteLine("-----------------------------------------");

            // Create purchase order PDF
            var pdf = CreateSamplePdf("Purchase Order #PO-2024-001",
                "Items: Office Supplies\nTotal: $5,000.00\n\nApproval Required: Manager, Finance");
            var inputPath = Path.Combine(OutputDir, "purchase-order-input.pdf");
            pdf.SaveAs(inputPath);
            Console.WriteLine($"  Created: purchase-order-input.pdf");

            // First signature (Manager)
            var pdfToSign1 = PdfDocument.FromFile(inputPath);
            var signature1 = new PdfSignature(CertificatePath, CertificatePassword)
            {
                SigningReason = "Manager Approval",
                SigningLocation = "Department A"
            };
            pdfToSign1.Sign(signature1);
            var managerApprovedPath = Path.Combine(OutputDir, "po-manager-approved.pdf");
            pdfToSign1.SaveAs(managerApprovedPath);
            Console.WriteLine($"  Created: po-manager-approved.pdf (1st signature)");

            // Second signature (Finance) - using incremental save
            var pdfToSign2 = PdfDocument.FromFile(managerApprovedPath);
            var signature2 = new PdfSignature(CertificatePath, CertificatePassword)
            {
                SigningReason = "Finance Approval",
                SigningLocation = "Finance Department"
            };
            pdfToSign2.Sign(signature2);
            var financeApprovedPath = Path.Combine(OutputDir, "po-finance-approved.pdf");
            pdfToSign2.SaveAs(financeApprovedPath);
            Console.WriteLine($"  Created: po-finance-approved.pdf (2nd signature)");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test9_VerifySignatures()
        {
            Console.WriteLine("Test 9: Verify Signatures");
            Console.WriteLine("-------------------------");

            // Load a signed document
            var signedPath = Path.Combine(OutputDir, "contract-signed.pdf");
            if (!File.Exists(signedPath))
            {
                Console.WriteLine("  Skipped: No signed document found");
                return;
            }

            var pdf = PdfDocument.FromFile(signedPath);
            bool isValid = pdf.VerifyPdfSignatures();

            Console.WriteLine($"  Document: contract-signed.pdf");
            Console.WriteLine($"  Signatures Valid: {isValid}");

            // Get signature details
            var signatures = pdf.GetVerifiedSignatures();
            Console.WriteLine($"  Number of Signatures: {signatures.Count}");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test10_DetectTampering()
        {
            Console.WriteLine("Test 10: Detect Tampered Documents");
            Console.WriteLine("-----------------------------------");

            // Load a signed document and verify
            var signedPath = Path.Combine(OutputDir, "contract-signed.pdf");
            if (!File.Exists(signedPath))
            {
                Console.WriteLine("  Skipped: No signed document found");
                return;
            }

            var pdf = PdfDocument.FromFile(signedPath);
            bool isValid = pdf.VerifyPdfSignatures();

            if (isValid)
            {
                Console.WriteLine("  Document has not been tampered with");
                Console.WriteLine("  All signatures are valid");
            }
            else
            {
                Console.WriteLine("  WARNING: Document may have been tampered with!");
                Console.WriteLine("  One or more signatures are invalid");
            }

            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void Test11_ComplianceSigning()
        {
            Console.WriteLine("Test 11: Compliance Signing");
            Console.WriteLine("---------------------------");

            var pdf = CreateSamplePdf("Compliance Document",
                "This document requires regulatory compliance certification.\n\n" +
                "Document ID: COMP-2024-001\n" +
                "Classification: Confidential\n" +
                "Retention Period: 7 years");
            var inputPath = Path.Combine(OutputDir, "compliance-document-input.pdf");
            pdf.SaveAs(inputPath);
            Console.WriteLine($"  Created: compliance-document-input.pdf");

            var pdfToSign = PdfDocument.FromFile(inputPath);
            var signature = new PdfSignature(CertificatePath, CertificatePassword)
            {
                SigningReason = "21 CFR Part 11 Compliance Certification",
                SigningLocation = "Quality Assurance Department",
                SigningContact = "compliance@company.com",
                SignatureDate = DateTime.UtcNow
                // For production compliance, add timestamp server:
                // TimestampHashAlgorithm = TimestampHashAlgorithms.SHA256,
                // TimeStampUrl = new Uri("http://timestamp.digicert.com")
            };

            pdfToSign.Sign(signature);
            var outputPath = Path.Combine(OutputDir, "compliance-document-certified.pdf");
            pdfToSign.SaveAs(outputPath);

            Console.WriteLine($"  Created: compliance-document-certified.pdf");
            Console.WriteLine($"  Reason: {signature.SigningReason}");
            Console.WriteLine($"  Location: {signature.SigningLocation}");
            Console.WriteLine("  Status: SUCCESS\n");
        }

        static void GenerateScreenshots()
        {
            Console.WriteLine("Generating PNG Screenshots");
            Console.WriteLine("--------------------------");

            // Generate screenshots for input PDFs (before signing)
            string[] inputPdfs = {
                "contract-input.pdf",
                "invoice-input.pdf",
                "agreement-input.pdf",
                "purchase-order-input.pdf",
                "compliance-document-input.pdf",
                "multi-page-contract-input.pdf"
            };

            foreach (var pdfName in inputPdfs)
            {
                var pdfPath = Path.Combine(OutputDir, pdfName);
                if (File.Exists(pdfPath))
                {
                    var pdf = PdfDocument.FromFile(pdfPath);
                    var pngName = pdfName.Replace(".pdf", ".png");
                    var pngPath = Path.Combine(OutputDir, pngName);
                    pdf.RasterizeToImageFiles(pngPath);
                    Console.WriteLine($"  Created: {pngName}");
                }
            }

            // Generate screenshot for multi-page last page
            var multiPagePath = Path.Combine(OutputDir, "multi-page-contract-input.pdf");
            if (File.Exists(multiPagePath))
            {
                var pdf = PdfDocument.FromFile(multiPagePath);
                // Get the last page as separate image
                var lastPagePng = Path.Combine(OutputDir, "multi-page-contract-input-last.png");
                var lastPageIndex = pdf.PageCount - 1;
                var images = pdf.ToBitmap();
                if (images.Length > lastPageIndex)
                {
                    images[lastPageIndex].SaveAs(lastPagePng);
                    Console.WriteLine($"  Created: multi-page-contract-input-last.png (page {lastPageIndex + 1})");
                }
            }

            Console.WriteLine("  Status: SUCCESS\n");
        }
    }
}
